<?php

/**
 * Copyright (c) 2020 Nexi Payments S.p.A.
 *
 * @author      iPlusService S.r.l.
 * @copyright   Copyright (c) 2020 Nexi Payments S.p.A. (https://ecommerce.nexi.it)
 * @license     GNU General Public License v3.0
 *
 * @category    Payment Module
 *
 * @version     8.0.0
 */

namespace Nexi\Redirect\ApplePay;

if (!defined('_PS_VERSION_')) {
    exit;
}

use NexiXPay;

use Nexi\Redirect\Settings;
use Nexi\Utility\CurrencyHelper;
use \Nexi\NPG\Redirect\NPG;
use Nexi\Redirect\ApplePay\Configuration as ApplePayConfiguration;
use Nexi\Utility\Helper;

class Payment
{
    /**
     * @var \NexiXPay
     */
    private $module;
    private $codTrans;
    private $context;

    public function __construct($module)
    {
        $this->module = $module;

        $nexixpay = new NexiXPay();
        $this->context = $nexixpay->getContext();
    }

    public function getApplePayConfiguration()
    {
        $data = [];
        $config = Settings::getConfiguration();
        $applePay = new ApplePayConfiguration($this->module);
        $currencyHelper = new CurrencyHelper;
        $context = $this->getThisContext();

        $data['config'] = $applePay->getConfiguration();
        $data['config']['NEXIXPAY_GATEWAY'] = $config['gateway'];
        $data['config']['NEXIXPAY_TEST_MODE'] = $config['test_mode'];
        $data['cards'] = $this->getMappedCards();
        $data['transactionInfo']['countryCode'] = $context->country->iso_code;
        $data['transactionInfo']['currencyCode'] = $context->currency->iso_code;
        $data['transactionInfo']['totalPrice'] = (string) $currencyHelper->getRoundedAmountNPG($context->smarty->tpl_vars['cart']->value['totals']['total_including_tax']['amount'], $data['transactionInfo']['currencyCode']);
        $data['lang'] = $context->language->iso_code;

        return $data;
    }

    public function getMappedCards()
    {
        $nexixpay = new NexiXPay();

        $cardsMapping = [
            'MAESTRO' => 'maestro',
            'MASTERCARD' => 'masterCard',
            'MC' => 'masterCard',
            'VISA' => 'visa',
            'JCB' => 'jcb',
            'AMEX' => 'amex',
        ];

        $availableMethods = $nexixpay->getSortedCardsQuery();

        $cards = [];

        foreach ($availableMethods as $availableMethod) {
            if (isset($cardsMapping[$availableMethod])) {
                $cards[] = $cardsMapping[$availableMethod];
            }
        }

        return $cards;
    }

    public function validateMerchant($request)
    {
        $applePay = new ApplePayConfiguration($this->module);

        $applePayConfig = $applePay->getConfiguration();

        $directory = _PS_MODULE_DIR_ . $this->module->name . "/upload";

        $extraOptions = [
            CURLOPT_TIMEOUT => 25,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSLVERSION => CURL_SSLVERSION_TLSv1_2,
            CURLOPT_SSLCERT => $directory . "/" . $applePayConfig['NEXIXPAY_APPLE_PAY_MERCHANT_IDENTIFIER_CERTIFICATE'],
            CURLOPT_SSLKEY => $directory . "/" . $applePayConfig['NEXIXPAY_APPLE_PAY_MERCHANT_IDENTIFIER_CERTIFICATE_KEY'],
            CURLOPT_CAINFO => $directory . "/" . $applePayConfig['NEXIXPAY_APPLE_PAY_MERCHANT_CA_ROOT_CERTIFICATE'],
        ];

        $payload = [
            'merchantIdentifier' => $applePayConfig['NEXIXPAY_APPLE_PAY_MERCHANT_ID'],
            'displayName' => $applePayConfig['NEXIXPAY_APPLE_PAY_MERCHANT_LABEL'],
            'initiative' => "web",
            'initiativeContext' => str_replace(["http://", "https://", "/"], "", $_SERVER['SERVER_NAME']),
        ];

        $requestUrl = json_decode($request, true);

        $curl = new \Nexi\Utility\CurlCall($requestUrl['validationUrl'], $payload, $extraOptions);

        $response = $curl->execCurl();

        return json_encode($response);
    }

    public function getPlaceOrderDeferredObjectXPay($request)
    {
        $nexixpay = new NexiXPay();

        $config = Settings::getConfiguration();

        $context = $this->getThisContext();

        $currencyHelper = new CurrencyHelper;

        $this->codTrans = \Tools::substr($context->cart->id . '-' . time(), 0, 30);

        $importo = (string) $currencyHelper->calculateAmountToMinUnitNPG($context->smarty->tpl_vars['cart']->value['totals']['total_including_tax']['amount'], $context->currency->iso_code);

        $timestamp = (time()) * 1000;

        $payload = [
            'apiKey' => $config['alias'],
            'codiceTransazione' => $this->codTrans,
            'importo' => $importo,
            'divisa' => '978',
            'applePay' => json_decode($request),
            'timeStamp' => $timestamp,
            'parametriAggiuntivi' => [
                'mail' => $context->smarty->tpl_vars['customer']->value['email'],
                'nome' => $context->smarty->tpl_vars['customer']->value['firstname'],
                'cognome' => $context->smarty->tpl_vars['customer']->value['lastname'],
            ],
        ];

        $payload['mac'] = Helper::hashHelper('apiKey=' . $payload['apiKey'] . 'codiceTransazione=' . $this->codTrans . 'importo=' . $importo . "divisa=" . $payload['divisa'] . "timeStamp=" . $payload['timeStamp'] . $config['mac_key'], 'sha1');

        $curl = new \Nexi\Utility\CurlCall($nexixpay->getXPayBaseUrl($config['test_mode']) . 'ecomm/api/paga/applePay', $payload);

        return [
            "response" => $curl->execCurl(),
            "request" => $payload
        ];
    }

    public function getPlaceOrderDeferredObjectNPG($request)
    {
        $nexixpay = new NexiXPay();

        $config = Settings::getConfiguration();

        $context = $this->getThisContext();

        $currencyHelper = new CurrencyHelper;

        $orderId = NPG::calculateOrderId($context->cart->id);

        $token = json_decode($request);

        $importo = (string) $currencyHelper->calculateAmountToMinUnitNPG($context->smarty->tpl_vars['cart']->value['totals']['total_including_tax']['amount'], $context->currency->iso_code);

        $payload = [
            'applePayPaymentData' => [
                'email' => $context->smarty->tpl_vars['customer']->value['email'],
                'token' => json_encode($token->token),
            ],
            'order' => [
                'amount' => $importo,
                'currency' => $context->currency->iso_code,
                'customerInfo' => [
                    'cardHolderEmail' => $context->smarty->tpl_vars['customer']->value['email'],
                    'cardholderName' => $context->smarty->tpl_vars['customer']->value['firstname'],
                ],
                'orderId' => $orderId,
            ],
            'paymentSession' => [
                'actionType' => 'PAY',
                'amount' => $importo,
                'language' => 'ita',
                'resultUrl' => $this->module->createModuleLink($this->module->name, 'npg-return', ['orderId' => $orderId]),
                'cancelUrl' => $this->module->createModuleLink($this->module->name, 'npg-cancel'),
            ],
        ];

        $curl = new \Nexi\Utility\CurlCall($nexixpay->getNPGBaseUrl($config['test_mode']) . 'orders/applepay', $payload);

        return $curl::exec_REST_CURL('POST', $nexixpay->getNPGBaseUrl($config['test_mode']) . 'orders/applepay', $payload, $config['api_key']);
    }

    public function saveOrder()
    {
        $config = \Nexi\Redirect\Settings::getConfiguration();

        $context = $this->getThisContext();

        $importo = $context->smarty->tpl_vars['cart']->value['totals']['total_including_tax']['amount'];

        $orderData = Db::getInstance()->getValue('
            SELECT `id_order`
            FROM `' . _DB_PREFIX_ . 'orders`
            WHERE `id_cart` = ' . (int) $context->cart->id
        );

        if ($orderData) {
            $order = new Order((int) $orderData);
        } else {
            $order = null;
        }

        if ($config['gateway'] === 'xpay') {
            $transId = $this->getCodTrans();
        } elseif ($config['gateway' === 'npg']) {
            $transId = $order->id;
        }

        $status = (int) \Configuration::get('PS_OS_PAYMENT');

        $this->module->validateOrder(
            $context->cart->id,
            $status,
            $importo,
            $this->module->displayName,
            null,
            ['transaction_id' => $transId],
            (int) $context->currency->id,
            false,
            $context->customer->secure_key
        );

        $url = $this->context->link->getPageLink(
            'order-confirmation',
            true,
            null,
            array(
                'id_cart' => (int) $context->cart->id,
                'id_module' => (int) $this->module->id,
                'id_order' => (int) $order->id,
                'key' => $context->customer->secure_key,
            )
        );

        return [
            'url' => $url
        ];
    }

    public function savePaymentInfo($data)
    {
        $context = $this->getThisContext();
        $oPI = new \PaymentInfo();
        $oPI->codTrans = $data['request']['codiceTransazione'];
        $oPI->idCart = $context->cart->id;
        $oPI->amount = $context->smarty->tpl_vars['cart']->value['totals']['total_including_tax']['amount'];
        $oPI->currency = $context->currency->iso_code;
        $oPI->brand = $data['response']['brand'];
        $oPI->result = $data['response']['esito'];
        $date = \DateTime::createFromFormat('YmdHis', (string) $data['response']['data'] . $data['response']['ora']);
        if ($date) {
            $oPI->date = $date->format('Y-m-d H:i:s');
        }
        $oPI->autCode = $data['response']['codiceAutorizzazione'];
        $oPI->firstName = $data['request']['parametriAggiuntivi']['nome'];
        $oPI->lastName = $data['request']['parametriAggiuntivi']['cognome'];
        $oPI->mail = $data['request']['parametriAggiuntivi']['mail'];
        $oPI->save();
    }

    private function getCodTrans()
    {
        return $this->codTrans;
    }

    private function getThisContext()
    {
        return $this->context;
    }
}