<?php

/**
 * Copyright (c) 2020 Nexi Payments S.p.A.
 *
 * @author      iPlusService S.r.l.
 * @copyright   Copyright (c) 2020 Nexi Payments S.p.A. (https://ecommerce.nexi.it)
 * @license     GNU General Public License v3.0
 *
 * @category    Payment Module
 *
 * @version     8.0.0
 */

namespace Nexi\Redirect\GooglePay;

if (!defined('_PS_VERSION_')) {
    exit;
}

use NexiXPay;

use Nexi\Redirect\Settings;
use Nexi\Utility\CurrencyHelper;
use Nexi\Utility\Logger;
use \Nexi\NPG\Redirect\NPG;
use Nexi\Redirect\GooglePay\Configuration as GooglePayConfiguration;
use Nexi\Utility\Helper;

class Payment
{

    /**
     * @var \NexiXPay
     */
    private $module;
    private $codTrans;
    private $context;

    public function __construct($module)
    {
        $this->module = $module;

        $nexixpay = new NexiXPay();
        $this->context = $nexixpay->getContext();
    }

    public function getGooglePayConfiguration()
    {
        $data = [];
        $config = Settings::getConfiguration();
        $nexixpay = new NexiXPay();
        $googlePay = new GooglePayConfiguration($this->module);
        $currencyHelper = new CurrencyHelper;
        $context = $this->getThisContext();

        $data['config'] = $googlePay->getConfiguration();
        $data['config']['NEXIXPAY_GATEWAY'] = $config['gateway'];
        $data['config']['NEXIXPAY_TEST_MODE'] = $config['test_mode'];
        $data['cards'] = $nexixpay->getSortedCardsQuery();
        $data['transactionInfo']['countryCode'] = $context->country->iso_code;
        $data['transactionInfo']['currencyCode'] = $context->currency->iso_code;
        $data['transactionInfo']['totalPrice'] = (string) $currencyHelper->getRoundedAmountNPG($context->smarty->tpl_vars['cart']->value['totals']['total_including_tax']['amount'], $data['transactionInfo']['currencyCode']);
        $data['lang'] = $context->language->iso_code;

        return $data;
    }

    public function finalizePaymentXPay($paymentData)
    {
        $response = $this->googlePayPaymentXPay($paymentData);

        $context = $this->getThisContext();

        $codTrans = $this->getCodTrans();

        $currencyHelper = new CurrencyHelper;

        $importo = (string) $currencyHelper->calculateAmountToMinUnitNPG($context->smarty->tpl_vars['cart']->value['totals']['total_including_tax']['amount'], $context->currency->iso_code);

        $amountToSave = $currencyHelper->fromMinUnitToAmountXPay($importo, $context->currency->iso_code);

        if ($response['esito'] === 'OK') {
            if ($response['html']) {
                $this->save3DSHtml($context->cart->id, $this->codTrans, $response['html'], $importo);

                $url = [
                    'url' => $this->module->createModuleLink($this->module->name, 'googlepay-load3DShtml', ['codTrans' => $codTrans]),
                ];
            } else {
                $this->saveXPayPaymentInfo(
                    $context->cart,
                    $codTrans,
                    $response['esito'],
                    $amountToSave,
                    null,
                    null,
                    $response['brand'],
                    $response['data'] . ' ' . $response['ora'],
                    $response['nazione'],
                    $response['codiceAutorizzazione'],
                    null,
                    $context->customer->firstname,
                    $context->customer->lastname,
                    $context->customer->email
                );

                $url = $this->saveOrder($context, $importo, $codTrans);
            }
        } else {
            $url = [
                'url' => $this->module->createModuleLink($this->module->name, 'cancellation'),
            ];
        }

        return $url;
    }

    private function googlePayPaymentXPay($paymentData)
    {
        $currencyHelper = new CurrencyHelper;

        $config = Settings::getConfiguration();

        $context = $this->getThisContext();

        $url = $this->module->createModuleLink($this->module->name, 'googlepay-response');

        $this->codTrans = \Tools::substr($context->cart->id . '-' . time(), 0, 30);

        $timestamp = (time()) * 1000;

        $importo = (string) $currencyHelper->calculateAmountToMinUnitNPG($context->smarty->tpl_vars['cart']->value['totals']['total_including_tax']['amount'], $context->currency->iso_code);

        $payload = [
            'apiKey' => $config['alias'],
            'codiceTransazione' => $this->codTrans,
            'importo' => $importo,
            'divisa' => "978",
            'googlePay' => json_encode($paymentData),
            'urlRisposta' => $url,
            'timeStamp' => $timestamp,
            'parametriAggiuntivi' => [
                'mail' => $context->smarty->tpl_vars['customer']->value['email'],
                'nome' => $context->smarty->tpl_vars['customer']->value['firstname'],
                'cognome' => $context->smarty->tpl_vars['customer']->value['lastname'],
            ]
        ];

        $payload['mac'] = Helper::hashHelper('apiKey=' . $payload['apiKey'] . 'codiceTransazione=' . $this->codTrans . 'importo=' . $importo . "divisa=" . $payload['divisa'] . "timeStamp=" . $payload['timeStamp'] . $config['mac_key'], 'sha1');

        $curl = new \Nexi\Utility\CurlCall($this->getXPayBaseUrl($config['test_mode']) . 'ecomm/api/paga/v2/pagaGooglePay', $payload);

        return $curl->execCurl();
    }

    public function finalizePaymentNPG($paymentData)
    {
        $response = $this->googlePayPaymentNPG($paymentData);

        $codTrans = $this->getCodTrans();

        $currencyhelper = new CurrencyHelper;

        $context = $this->getThisContext();

        $config = Settings::getConfiguration();

        $importo = (string) $currencyhelper->calculateAmountToMinUnitNPG($context->smarty->tpl_vars['cart']->value['totals']['total_including_tax']['amount'], $context->currency->iso_code);

        if (isset($response['response']['paymentFlow']) && $response['response']['paymentFlow'] == 'PAN_ONLY' && isset($response['response']['iframe']) && $response['response']['iframe']['state'] == 'GDI_VERIFICATION') {
            $this->saveIframeSessionId($context->cart->id, $codTrans, $response['response']['iframe']['sessionId'], $importo);

            $response = $response['response'];

            if ($config['test_mode'] == 1) {
                $response['sdkLink'] = 'https://stg-ta.nexigroup.com/monetaweb/resources/hfsdk.js';
            } else {
                $response['sdkLink'] = 'https://xpay.nexigroup.com/monetaweb/resources/hfsdk.js';
            }

            $response['controllerUrlGooglePay'] = $this->module->createModuleLink($this->module->name, 'googlepay-manageSdk', ['codTrans' => $codTrans]);
        } else {
            if ($response['response']['operation']['operationResult'] == 'EXECUTED') {
                try {
                    $oI = new \OrderInfo();

                    if (\NexiXPay::isXPayBuild() && $oI->cartIdAndOrderIdExists(NPG::getCartId($response['response']['operation']['orderId']), $response['response']['operation']['orderId'])) {
                        $oI->updateExistingPaymentInfo(
                            NPG::getCartId($response['response']['operation']['orderId']),
                            $response['response']['operation']['orderId'],
                            $response['response']['securityToken']
                        );
                    } else {
                        $oI->saveInfo(
                            NPG::getCartId($response['response']['operation']['orderId']),
                            $response['response']['operation']['orderId'],
                            $response['response']['securityToken']
                        );
                    }
                } catch (\Exception $exc) {
                    Logger::logExceptionError($exc);

                    throw new \Exception($exc->getMessage());
                }

                $response = $this->saveOrder($context, $importo, $codTrans);
            } else {
                Logger::log(__FUNCTION__ . ': KO - Request: ' . $response['response']['errors'][0]['description'], 1);

                $response = [
                    'url' => $this->module->createModuleLink($this->module->name, 'npg-cancel'),
                ];
            }
        }

        return $response;
    }

    private function googlePayPaymentNPG($paymentData)
    {
        $currencyHelper = new CurrencyHelper;

        $context = $this->getThisContext();

        $this->codTrans = \Tools::substr($context->cart->id . '-' . time(), 0, 30);

        $importo = (string) $currencyHelper->calculateAmountToMinUnitNPG($context->smarty->tpl_vars['cart']->value['totals']['total_including_tax']['amount'], $context->currency->iso_code);

        $orderId = NPG::calculateOrderId($context->cart->id);

        $currency = CurrencyHelper::getCurrencyISOCode($context->cart->id_currency);

        $config = Settings::getConfiguration();

        $payload = [
            'order' => [
                'orderId' => (string) $orderId,
                'amount' => (string) $importo,
                'currency' => $currency,
                'description' => 'PS Order: ' . $context->cart->id,
                'customField' => 'Prestashop ' . \Tools::substr(_PS_VERSION_, 0, 3) . '.x - ' . $this->module->name . ' ' . $this->module->version,
                'customerId' => $context->cart->id_customer,
            ],
            'paymentSession' => [
                'actionType' => 'PAY',
                'amount' => (string) $importo,
                'recurrence' => [
                    'action' => NO_RECURRING,
                ],
                'exemptions' => 'NO_PREFERENCE',
                'language' => Settings::getPaymentGatewayLanguage($context),
                'resultUrl' => $this->module->createModuleLink($this->module->name, 'npg-return', ['orderId' => $orderId]),
                'cancelUrl' => $this->module->createModuleLink($this->module->name, 'npg-cancel'),
            ],
            'googlePayPaymentData' => [
                'apiVersion' => (int) $paymentData['apiVersion'],
                'apiVersionMinor' => (int) $paymentData['apiVersionMinor'],
                'email' => $context->smarty->tpl_vars['customer']->value['email'],
                'paymentMethodData' => [
                    'cardInfo' => [
                        'cardDetails' => $paymentData['paymentMethodData']['info']['cardDetails'],
                        'cardNetwork' => $paymentData['paymentMethodData']['info']['cardNetwork'],
                    ],
                    'description' => $paymentData['paymentMethodData']['description'],

                    'tokenizationData' => [
                        'token' => $paymentData['paymentMethodData']['tokenizationData']['token'],
                        'type' => $paymentData['paymentMethodData']['tokenizationData']['type']
                    ],
                    'type' => $paymentData['paymentMethodData']['type']
                ],
            ]
        ];

        $curl = new \Nexi\Utility\CurlCall($this->getNPGBaseUrl($config['test_mode']) . 'orders/googlepay', $payload);

        return $curl::exec_REST_CURL('POST', $this->getNPGBaseUrl($config['test_mode']) . 'orders/googlepay', $payload, $config['api_key']);
    }

    public function paga3ds($xpayNonce)
    {
        $config = Settings::getConfiguration();

        $order = $this->getOrderIdsFromDB();

        $cart = new \Cart($order['cart_id']);
        $customer = new \Customer($cart->id_customer);

        $timestamp = (time()) * 1000;

        $payload = [
            'apiKey' => $config['alias'],
            'codiceTransazione' => $order['trans_id'],
            'importo' => $order['importo'],
            'divisa' => "978",
            'xpayNonce' => $xpayNonce,
            'timeStamp' => $timestamp,
            'parametriAggiuntivi' => [
                'mail' => $customer->email,
                'nome' => $customer->firstname,
                'cognome' => $customer->lastname,
                'TCONTAB' => $config['accounting']
            ],
        ];

        $payload['mac'] = Helper::hashHelper('apiKey=' . $payload['apiKey'] . 'codiceTransazione=' . $order['trans_id'] . 'importo=' . $order['importo'] . "divisa=" . $payload['divisa'] . "xpayNonce=" . $xpayNonce . "timeStamp=" . $payload['timeStamp'] . $config['mac_key'], 'sha1');

        $curl = new \Nexi\Utility\CurlCall($this->getXPayBaseUrl($config['test_mode']) . 'ecomm/api/paga/paga3DS', $payload);

        return $curl->execCurl();
    }

    public function googlePayResponseResult()
    {
        $infos = $this->getOrderIdsFromDB();
        $cart = new \Cart($infos['cart_id']);
        $customer = new \Customer($cart->id_customer);
        $currency = new \Currency($cart->id_currency);

        $importo = CurrencyHelper::fromMinUnitToAmountXPay($infos['importo'], currency: $currency->iso_code);

        $status = \Configuration::get('PS_OS_PAYMENT');

        $this->module->validateOrder(
            $infos['cart_id'],
            $status,
            (float) $importo,
            $this->module->displayName,
            null,
            ['transaction_id' => $infos['trans_id']],
            $currency->id,
            false,
            $customer->secure_key
        );

        $id_order = \Db::getInstance()->getValue('
        SELECT `id_order`
        FROM `' . _DB_PREFIX_ . 'orders`
        WHERE `id_cart` = ' . $infos['cart_id']
        );

        if ($id_order) {
            $order = new \Order($id_order);
        } else {
            $order = null;
        }

        $redirectUrl = 'index.php?controller=order-confirmation'
            . '&id_cart=' . $cart->id
            . '&id_module=' . $this->module->id
            . '&id_order=' . ($order ? $order->id : '')
            . '&key=' . $customer->secure_key;

        \Tools::redirect($redirectUrl);
    }

    public function saveOrder($context, $importo, $codTrans = null)
    {
        $currencyHelper = new CurrencyHelper;

        $importo = $currencyHelper->fromMinUnitToAmountXPay($importo, $context->currency->iso_code);

        $order = new \Order(\Order::getOrderByCartId($context->cart->id));

        $this->module->validateOrder(
            $context->cart->id,
            (int) \Configuration::get('PS_OS_PAYMENT'),
            $importo,
            $this->module->displayName,
            null,
            ['transaction_id' => $codTrans],
            (int) $context->currency->id,
            false,
            $context->customer->secure_key
        );

        $orderData = Db::getInstance()->getValue('
            SELECT `id_order`
            FROM `' . _DB_PREFIX_ . 'orders`
            WHERE `id_cart` = ' . (int) $context->cart->id
        );

        if ($orderData) {
            $order = new Order((int) $orderData);
        } else {
            $order = null;
        }

        $url = $this->context->link->getPageLink(
            'order-confirmation',
            true,
            null,
            array(
                'id_cart' => (int) $context->cart->id,
                'id_module' => (int) $this->module->id,
                'id_order' => (int) $order->id,
                'key' => $context->customer->secure_key,
            )
        );

        return [
            'url' => $url
        ];
    }

    private function getNPGBaseUrl($isTest)
    {
        if ($isTest === true) {
            return 'https://stg-ta.nexigroup.com/api/phoenix-0.0/psp/api/v1/';
        } elseif ($isTest === false) {
            return 'https://xpay.nexigroup.com/api/phoenix-0.0/psp/api/v1/';
        }
    }

    private function getXPayBaseUrl($isTest)
    {
        $paymentGateway = new \Nexi\XPay\Redirect\Parameters();

        if ($isTest === true) {
            return $paymentGateway->urlIntegration;
        } elseif ($isTest === false) {
            return $paymentGateway->urlProduction;
        }
    }

    public function getThisContext()
    {
        return $this->context;
    }
    private function getCodTrans()
    {
        return $this->codTrans;
    }

    private function save3DSHtml($cartId, $codTrans, $html, $importo)
    {
        //base64 to bypass injection elements
        $query = " REPLACE INTO `" . _DB_PREFIX_ . "nexi_googlepay` (cart_id, trans_id, html, importo) VALUES (
                '" . pSQL($cartId) . "', '" . pSQL($codTrans) . "', '" . pSQL(base64_encode($html), true) . "', '" . pSQL($importo) . "'
            );";

        $nexixpay = new NexiXPay();

        $res = $nexixpay->execQuery($query);

        if ($res === false) {
            return false;
        }
    }
    public function get3dsHtml($codTrans)
    {
        $query = '  SELECT html
            FROM `' . _DB_PREFIX_ . "nexi_googlepay`
            WHERE trans_id = '" . pSQL($codTrans) . "'";

        $nexixpay = new NexiXPay();

        return base64_decode($nexixpay->execSelectQuery($query)[0]['html']);
    }

    private function saveIframeSessionId($cartId, $codTrans, $sessionId, $importo)
    {
        $query = " REPLACE INTO `" . _DB_PREFIX_ . "nexi_googlepay` (cart_id, trans_id, session_id, importo) VALUES (
                '" . pSQL($cartId) . "', '" . pSQL($codTrans) . "', '" . pSQL($sessionId) . "', '" . pSQL($importo) . "'
            );";

        $nexixpay = new NexiXPay();

        $res = $nexixpay->execQuery($query);

        if ($res === false) {
            return false;
        }
    }

    public function getIframeSessionId($codTrans)
    {
        $query = '  SELECT `session_id`
            FROM `' . _DB_PREFIX_ . "nexi_googlepay`
            WHERE trans_id = '" . pSQL($codTrans) . "'";

        $nexixpay = new NexiXPay();

        return $nexixpay->execSelectQuery($query)[0]['session_id'];
    }

    public function getOrderIdsFromDB()
    {
        $context = $this->getThisContext();

        $query = '  SELECT *
            FROM `' . _DB_PREFIX_ . "nexi_googlepay`
            WHERE `cart_id` = '" . pSQL($context->cart->id) .
            "' ORDER BY created_at DESC";

        $nexixpay = new NexiXPay();

        $data = $nexixpay->execSelectQuery($query);

        return [
            'cart_id' => $context->cart->id,
            'trans_id' => $data[0]['trans_id'],
            'importo' => $data[0]['importo'],
            'session_id' => $data[0]['session_id']
        ];
    }

    public function saveXPayPaymentInfo(
        $cart,
        $codTrans,
        $esito,
        $importo,
        $pan = null,
        $scadenza_pan = null,
        $brand = null,
        $data = null,
        $nazione = null,
        $codice_autorizzazione = null,
        $num_contratto = null,
        $nome = null,
        $cognome = null,
        $mail = null
    ) {
        $oPI = new \PaymentInfo();

        $currency_order = new \Currency($cart->id_currency);

        $oPI->codTrans = $codTrans;
        $oPI->idCart = $cart->id;
        $oPI->amount = $importo;
        $oPI->currency = $currency_order->iso_code;
        $oPI->brand = $brand;
        $oPI->result = $esito;
        $oPI->autCode = $codice_autorizzazione;
        $oPI->pan = $pan;
        $oPI->exp = $scadenza_pan;
        $oPI->nationality = $nazione;
        $oPI->contractNumber = $num_contratto;
        $oPI->firstName = $nome;
        $oPI->lastName = $cognome;
        $oPI->mail = $mail;

        $date = \DateTime::createFromFormat('d/m/Y H:i:s', $data);

        if ($date === false) {
            $date = new \DateTime();
        }

        $oPI->date = $date->format('Y-m-d H:i:s');

        $oPI->save();
    }
}