/**
 * Copyright (c) 2020 Nexi Payments S.p.A.
 *
 * @author      iPlusService S.r.l.
 * @copyright   Copyright (c) 2020 Nexi Payments S.p.A. (https://ecommerce.nexi.it)
 * @license     GNU General Public License v3.0
 *
 * @category    Payment Module
 *
 * @version     8.0.0
 */
let paymentsClient = null;

const baseRequest = {
  apiVersion: 2,
  apiVersionMinor: 0,
};

function getXPayGooglePaymentDataRequest(configuration) {
  const paymentDataRequest = Object.assign({}, baseRequest);
  if (configuration.config.NEXIXPAY_GATEWAY === "xpay") {
    paymentDataRequest.allowedPaymentMethods = [
      Object.assign({}, getXPayGooglePayBaseCardPaymentMethod(configuration), {
        tokenizationSpecification: {
          type: "PAYMENT_GATEWAY",
          parameters: {
            gateway: "nexi",
            gatewayMerchantId:
              configuration.config.NEXIXPAY_GOOGLE_PAY_GATEWAY_MERCHANT_ID,
          },
        },
      }),
    ];
  } else if (configuration.config.NEXIXPAY_GATEWAY === "npg") {
    paymentDataRequest.allowedPaymentMethods = [
      Object.assign({}, getXPayGooglePayBaseCardPaymentMethod(configuration), {
        tokenizationSpecification: {
          type: "PAYMENT_GATEWAY",
          parameters: {
            gateway: "nexigtw",
            gatewayMerchantId:
              configuration.config.NEXIXPAY_GOOGLE_PAY_GATEWAY_MERCHANT_ID,
          },
        },
      }),
    ];
  }

  paymentDataRequest.transactionInfo = {
    countryCode: configuration.transactionInfo.countryCode,
    currencyCode: configuration.transactionInfo.currencyCode,
    totalPriceStatus: "FINAL",
    totalPrice: configuration.transactionInfo.totalPrice,
  };

  paymentDataRequest.merchantInfo = {
    merchantId: configuration.config.NEXIXPAY_GOOGLE_PAY_MERCHANT_ID,
    merchantName: configuration.config.NEXIXPAY_GOOGLE_PAY_MERCHANT_NAME,
  };

  return paymentDataRequest;
}

function onXPayGooglePayLoaded() {
  getXPayGooglePayConfiguration().then(function (configuration) {
    if (
      configuration.config.NEXIXPAY_TEST_MODE == true &&
      configuration.config.NEXIXPAY_GATEWAY === "xpay"
    ) {
      paymentsClient = new google.payments.api.PaymentsClient({
        environment: "TEST",
      });
    } else {
      paymentsClient = new google.payments.api.PaymentsClient({
        environment: "PRODUCTION",
      });
    }

    const isReadyToPayRequest = Object.assign({}, baseRequest, {
      allowedPaymentMethods: [
        getXPayGooglePayBaseCardPaymentMethod(configuration),
      ],
    });

    paymentsClient
      .isReadyToPay(isReadyToPayRequest)
      .then(function (response) {
        if (response.result) {
          addXPayGooglePayButton(configuration);
          prefetchXPayGooglePaymentData(configuration);
        }
      })
      .catch(function (err) {
        console.error(err);
      });
  });
}

function addXPayGooglePayButton(configuration) {
  const button = paymentsClient.createButton({
    onClick: function () {
      onXPayGooglePaymentButtonClicked(configuration);
    },
    buttonSizeMode: "fill",
    buttonColor: configuration.config.NEXIXPAY_GOOGLE_PAY_BUTTON_COLOR,
    buttonType: configuration.config.NEXIXPAY_GOOGLE_PAY_BUTTON_TYPE,
    buttonLocale: configuration.lang,
  });

  document.getElementById("google-pay-button-container").appendChild(button);
}

function prefetchXPayGooglePaymentData(configuration) {
  const paymentDataRequest = getXPayGooglePaymentDataRequest(configuration);
  paymentDataRequest.transactionInfo = {
    countryCode: configuration.transactionInfo.countryCode,
    currencyCode: configuration.transactionInfo.currencyCode,
    totalPriceStatus: "FINAL",
    totalPrice: configuration.transactionInfo.totalPrice,
  };

  paymentsClient.prefetchPaymentData(paymentDataRequest);
}

function onXPayGooglePaymentButtonClicked(configuration) {
  const paymentDataRequest = getXPayGooglePaymentDataRequest(configuration);

  paymentsClient
    .loadPaymentData(paymentDataRequest)
    .then(function (paymentData) {
      processXPayGooglePayPayment(paymentData);
    })
    .catch(function (err) {
      console.error(err);
    });
}

function processXPayGooglePayPayment(paymentData) {
  paymentToken = paymentData.paymentMethodData.tokenizationData.token;
  finalizeXPayGooglePayPayment(paymentData);
}

function getXPayGooglePayBaseCardPaymentMethod(configuration) {
  const allowedCardNetworks = [];

  configuration.cards.forEach(function (card) {
    if (card === "MC") {
      allowedCardNetworks.push("MASTERCARD");
    } else if (card === "UPI") {
    } else if (card === "Maestro") {
    } else {
      allowedCardNetworks.push(card);
    }
  });

  return {
    type: "CARD",
    parameters: {
      allowedAuthMethods: ["PAN_ONLY", "CRYPTOGRAM_3DS"],
      allowedCardNetworks: allowedCardNetworks,
    },
  };
}

function getXPayGooglePayConfiguration() {
  return new Promise(function (resolve, reject) {
    jQuery.ajax({
      type: "GET",
      data: {
        action: "getXPayGooglePayConfiguration",
      },
      url: $("#nexi_googlepay_url").val(),
      success: function (response) {
        resolve(JSON.parse(response, true));
      },
      error: function (xhr, ajaxOptions, thrownError) {
        console.error(
          thrownError + "\r\n" + xhr.statusText + "\r\n" + xhr.responseText
        );
      },
    });
  });
}

function loadGooglePaySdk(sdkLink, onGooglePaySdkLoad) {
  if (window.Build === undefined) {
    const imported = document.createElement("script");
    imported.onload = onGooglePaySdkLoad;
    imported.src = sdkLink;
    imported.type = "text/javascript";

    document.head.appendChild(imported);
  } else {
    onGooglePaySdkLoad();
  }
}

function finalizeXPayGooglePayPayment(paymentData) {
  return new Promise(function (resolve, reject) {
    jQuery.ajax({
      type: "POST",
      data: {
        action: "finalizeXPayGooglePayPayment",
        body: paymentData,
      },
      url: $("#nexi_googlepay_url").val(),
      success: function (response) {
        responseObj = JSON.parse(response, true);

        if (responseObj.url) {
          window.location.href = responseObj.url;
        } else {
          loadGooglePaySdk(responseObj.sdkLink, function () {
            new Build({
              onBuildFlowStateChange: function (evtData) {
                if (evtData.event === "BUILD_FLOW_STATE_CHANGE") {
                  if (
                    evtData.data.state === "REDIRECTED_TO_EXTERNAL_DOMAIN" ||
                    evtData.data.state === "PAYMENT_COMPLETE"
                  ) {
                    window.location.replace(responseObj.controllerUrlGooglePay);
                  }
                }
              },
            });
          });

          document.getElementById("google-pay-button-container").innerHTML = "";

          document.getElementById(
            "google-pay-button-container"
          ).style.visibility = "hidden";

          for (var i = 0; i < responseObj.iframe.fields.length; i++) {
            var iframe = document.createElement("iframe");

            iframe.src = responseObj.iframe.fields[i].src;

            document
              .getElementById("google-pay-button-container")
              .appendChild(iframe);
          }

          document.getElementById("google-pay-loader").style.visibility =
            "visible";
        }
      },

      error: function (xhr, ajaxOptions, thrownError) {
        console.error(
          thrownError + "\r\n" + xhr.statusText + "\r\n" + xhr.responseText
        );
      },
    });
  });
}
