
<?php

/**
 * Importa prodotti dal file Danea XML creando/aggiornando:
 *  - Associazioni a Categorie (Category/Subcategory) sotto Home
 *  - Associazione Marca (Manufacturer) da ProducerName
 *  - Associazioni Funzionalità (Features) da CustomField1..4 nel formato "Nome | Valori"
 *  - Quantità di magazzino (AvailableQty) via StockAvailable
 *  - GTIN/EAN (Barcode) forzato così com'è in product.ean13 (senza controlli)
 *  - Prezzo IVA esclusa (NetPrice1) -> Product::price (netto)
 *  - Classe IVA (Tax Rules Group) forzata per tutti i prodotti (SB_DANEA_TAX_RULE_GROUP_ID)
 *
 * Compatibile con PrestaShop 9 + PHP 8.3
 */
function sb_danea_import_products($module, $controller)
{
    // Percorso XML: prima da config del controller, poi fallback nella cartella tmp del modulo
    $xml_path = (string)Configuration::get('SB_DANEA_LAST_XML');
    if (!$xml_path) {
        $xml_path = _PS_MODULE_DIR_ . 'spiderbyte_danea/tmp/danea_last.xml';
    }
    if (!file_exists($xml_path)) {
        return '<div class="alert alert-danger">File XML non trovato: ' . htmlspecialchars($xml_path) . '</div>';
    }

    libxml_use_internal_errors(true);
    $xml = simplexml_load_file($xml_path);
    if (!$xml) {
        return '<div class="alert alert-danger">Errore nella lettura del file XML</div>';
    }

    $id_lang = 1; // Italiano
    $id_shop = (int)Context::getContext()->shop->id;
    $id_home = 2; // Categoria Home

    // === Forzatura IVA per tutti i prodotti (obbligatoria) ===
    $id_trg = (int)Configuration::get('SB_DANEA_TAX_RULE_GROUP_ID');
    if ($id_trg < 0) { $id_trg = 0; }
    if ($id_trg <= 0) {
        return '<div class="alert alert-danger">
            Nessun gruppo IVA selezionato. Imposta il "Tax Rules Group" nel modulo
            (chiave SB_DANEA_TAX_RULE_GROUP_ID) e riprova.
        </div>';
    }

    $created = 0; $updated = 0;
    $assoc_cat = 0; $assoc_feat = 0; $assoc_brand = 0;
    $qty_set = 0; $ean_set = 0; $price_set = 0; $tax_set = 0;

    // Cache per ridurre query ripetute
    $cache_categories = []; // key: parent|||name => id_category
    $cache_features   = []; // key: feature name => id_feature
    $cache_feat_vals  = []; // key: feature|||value => id_feature_value
    $cache_brands     = []; // key: brand name => id_manufacturer

    foreach ($xml->Products->Product as $prod) {

        // Identificazione prodotto: priorità Code -> Barcode -> InternalID
        $code  = trim((string)$prod->Code);
        $ean   = trim((string)$prod->Barcode);     // usato anche come reference se manca Code
        $iid   = trim((string)$prod->InternalID);

        $reference = $code !== '' ? $code : ($ean !== '' ? $ean : $iid);
        if ($reference === '') { continue; }

        // Cerca per reference
        $id_product = (int)Db::getInstance()->getValue(
            'SELECT id_product FROM '._DB_PREFIX_.'product WHERE reference = "'.pSQL($reference).'"'
        );

        if ($id_product) {
            $product = new Product($id_product, false, $id_lang);
            $updated++;
        } else {
            // Crea scheletro minimo (per poter associare tutto il resto)
            $name = trim((string)$prod->Description);
            if ($name === '') { $name = $reference; }
            $slug = Tools::str2url($name);

            $product = new Product();
            $product->active = 1;
            $product->reference = $reference;
            $product->id_category_default = $id_home; // aggiornato sotto se serve
            $product->name = [$id_lang => $name];
            $product->link_rewrite = [$id_lang => $slug];
            $product->price = 0; // netto
            $product->id_tax_rules_group = (int)$id_trg; // IVA forzata da subito

            $product->add();

            $id_product = (int)$product->id;
            $created++;
            $tax_set++; // conteggio assegnazione IVA su nuovi
        }

        $dirty = false; // per update batch unico

        // ===== Marca (Manufacturer) =====
        $brand_name = isset($prod->ProducerName) ? trim((string)$prod->ProducerName) : '';
        if ($brand_name !== '') {
            if (!isset($cache_brands[$brand_name])) {
                $id_man = (int)Db::getInstance()->getValue(
                    'SELECT id_manufacturer FROM '._DB_PREFIX_.'manufacturer WHERE name = "'.pSQL($brand_name).'"'
                );
                if (!$id_man) {
                    $m = new Manufacturer();
                    $m->name = $brand_name;
                    $m->active = 1;
                    $m->add();
                    $id_man = (int)$m->id;
                }
                $cache_brands[$brand_name] = $id_man;
            }
            $id_manufacturer = (int)$cache_brands[$brand_name];
            if ((int)$product->id_manufacturer !== $id_manufacturer) {
                $product->id_manufacturer = $id_manufacturer;
                $dirty = true;
                $assoc_brand++;
            }
        }

        // ===== Categorie =====
        $cat_name = trim((string)$prod->Category);
        $sub_name = isset($prod->Subcategory) ? trim((string)$prod->Subcategory) : '';

        $ids_to_add = [];
        if ($cat_name !== '') {
            $id_cat = ensureCategoryPS9_cached($cat_name, $id_home, $id_lang, $id_shop, $cache_categories);
            if ($id_cat) { $ids_to_add[] = $id_cat; }
            if ($sub_name !== '') {
                $id_sub = ensureCategoryPS9_cached($sub_name, $id_cat, $id_lang, $id_shop, $cache_categories);
                if ($id_sub) { $ids_to_add[] = $id_sub; }
                $default_cat = $id_sub;
            } else {
                $default_cat = $id_cat;
            }
        } else {
            $default_cat = $id_home;
        }

        if ((int)$product->id_category_default !== (int)$default_cat) {
            $product->id_category_default = (int)$default_cat;
            $dirty = true;
        }

        // Aggiungi associazioni categoria senza rimuovere le esistenti
        if (!empty($ids_to_add)) {
            $existing = Db::getInstance()->executeS(
                'SELECT id_category FROM '._DB_PREFIX_.'category_product WHERE id_product='.(int)$id_product
            );
            $have = [];
            foreach ($existing as $row) { $have[(int)$row['id_category']] = true; }
            foreach ($ids_to_add as $cid) {
                if (!isset($have[(int)$cid])) {
                    Db::getInstance()->insert('category_product', [
                        'id_category' => (int)$cid,
                        'id_product'  => (int)$id_product,
                        'position'    => 0,
                    ]);
                    $assoc_cat++;
                }
            }
        }

        // ===== Prezzo (NetPrice1) — netto =====
        if (isset($prod->NetPrice1)) {
            $net = (float)str_replace(',', '.', (string)$prod->NetPrice1);
            if ($net < 0) { $net = 0; }
            if ((float)$product->price !== $net) {
                $product->price = $net;
                $dirty = true;
                $price_set++;
            }
        }

        // ===== Forza sempre il Tax Rules Group selezionato =====
        if ((int)$product->id_tax_rules_group !== (int)$id_trg) {
            $product->id_tax_rules_group = (int)$id_trg;
            $dirty = true;
            $tax_set++;
        }

        // Salva modifiche prima di forzare il GTIN a DB (così evitiamo la validazione del model su ean13)
        if ($dirty) {
            $product->update();
            $dirty = false;
        }

        // ===== GTIN/EAN: forza <Barcode> così com'è in product.ean13 (senza controlli) =====
        $barcode_raw = isset($prod->Barcode) ? (string)$prod->Barcode : '';
        $barcode = trim($barcode_raw);
        if ($barcode !== '') {
            Db::getInstance()->update(
                'product',
                ['ean13' => pSQL($barcode)],
                'id_product='.(int)$id_product
            );
            $ean_set++;
        }

        // ===== Quantità (AvailableQty) via StockAvailable =====
        if (isset($prod->AvailableQty)) {
            $qty_raw = trim((string)$prod->AvailableQty);
            $qty = (int)floor((float)str_replace(',', '.', $qty_raw));
            if ($qty < 0) { $qty = 0; }
            StockAvailable::setQuantity((int)$product->id, 0, (int)$qty, (int)$id_shop);
            $qty_set++;
        }

        // ===== Funzionalità (Features) da CustomField1..4: "Nome | Valore1, Val2; Val3" =====
        $pairs = extractFeaturePairsFromCustomFields($prod);
        if (!empty($pairs)) {
            foreach ($pairs as $label => $values) {
                // Assicura feature
                if (!isset($cache_features[$label])) {
                    $id_feature = (int)Db::getInstance()->getValue(
                        'SELECT f.id_feature FROM '._DB_PREFIX_.'feature f '
                        . 'INNER JOIN '._DB_PREFIX_.'feature_lang fl ON (fl.id_feature=f.id_feature AND fl.id_lang='.(int)$id_lang.') '
                        . 'WHERE fl.name = "'.pSQL($label).'"'
                    );
                    if (!$id_feature) {
                        $f = new Feature();
                        $f->position = 0;
                        $f->name = [$id_lang => $label];
                        $f->add();
                        $id_feature = (int)$f->id;
                    }
                    $cache_features[$label] = $id_feature;
                }
                $id_feature = (int)$cache_features[$label];

                // Un solo valore per feature (se arrivano più valori, prevale l'ultimo)
                foreach ($values as $value) {
                    if ($value === '') { continue; }
                    $kv = $label.'|||'.$value;
                    if (!isset($cache_feat_vals[$kv])) {
                        $id_val = (int)Db::getInstance()->getValue(
                            'SELECT fv.id_feature_value FROM '._DB_PREFIX_.'feature_value fv '
                            . 'INNER JOIN '._DB_PREFIX_.'feature_value_lang fvl ON (fvl.id_feature_value=fv.id_feature_value AND fvl.id_lang='.(int)$id_lang.') '
                            . 'WHERE fv.id_feature='.(int)$id_feature.' AND fvl.value = "'.pSQL($value).'"'
                        );
                        if (!$id_val) {
                            $fv = new FeatureValue();
                            $fv->id_feature = $id_feature;
                            $fv->custom = 0;
                            $fv->value = [$id_lang => $value];
                            $fv->add();
                            $id_val = (int)$fv->id;
                        }
                        $cache_feat_vals[$kv] = $id_val;
                    }
                    $id_value = (int)$cache_feat_vals[$kv];

                    // Unicità per feature: rimuovi ed inserisci
                    Db::getInstance()->delete('feature_product', 'id_product='.(int)$id_product.' AND id_feature='.(int)$id_feature);
                    Db::getInstance()->insert('feature_product', [
                        'id_feature'        => (int)$id_feature,
                        'id_product'        => (int)$id_product,
                        'id_feature_value'  => (int)$id_value,
                    ]);
                    $assoc_feat++;
                }
            }
        }
    }

    return '<div class="alert alert-success">Import prodotti completato.<br />'
        . $created . ' prodotti creati, ' . $updated . ' aggiornati.<br />'
        . $assoc_cat . ' associazioni categoria, ' . $assoc_brand . ' marche associate, ' . $assoc_feat . ' feature associate.<br />'
        . $qty_set . ' quantità aggiornate, ' . $ean_set . ' GTIN impostati, ' . $price_set . ' prezzi aggiornati, ' . $tax_set . ' classi IVA impostate.</div>';
}

/**
 * Crea (se necessario) e restituisce l'id della categoria $name sotto $parent.
 * Usa cache passata per minimizzare query. Crea slug con Tools::str2url.
 */
function ensureCategoryPS9_cached($name, $parent, $id_lang, $id_shop, array &$cache_categories)
{
    $key = (int)$parent.'|||'.$name;
    if (isset($cache_categories[$key])) {
        return (int)$cache_categories[$key];
    }

    $id = (int)Db::getInstance()->getValue(
        'SELECT c.id_category FROM '._DB_PREFIX_.'category c '
        .'INNER JOIN '._DB_PREFIX_.'category_lang cl ON (cl.id_category=c.id_category AND cl.id_lang='.(int)$id_lang.' AND cl.id_shop='.(int)$id_shop.') '
        .'WHERE c.id_parent='.(int)$parent.' AND cl.name = "'.pSQL($name).'"'
    );
    if ($id) {
        $cache_categories[$key] = $id;
        return $id;
    }

    $slug = Tools::str2url($name);
    $cat = new Category();
    $cat->active = 1;
    $cat->id_parent = (int)$parent;
    $cat->name = [$id_lang => $name];
    $cat->link_rewrite = [$id_lang => $slug];
    $cat->add();
    $cat->addShop($id_shop);

    $cache_categories[$key] = (int)$cat->id;
    return (int)$cat->id;
}

/**
 * Estrae le coppie Caratteristica => [Valori] da CustomField1..4 nel formato
 *   Nome | Valore1, Valore2; Valore3
 */
function extractFeaturePairsFromCustomFields($prod)
{
    $pairs = [];
    for ($i=1; $i<=4; $i++) {
        $field = 'CustomField'.$i;
        if (!isset($prod->{$field})) { continue; }
        $raw = trim((string)$prod->{$field});
        if ($raw === '') { continue; }
        $parts = explode('|', $raw, 2);
        if (count($parts) < 2) { continue; }
        $label = trim($parts[0]);
        $values_str = trim($parts[1]);
        if ($label === '' || $values_str === '') { continue; }
        // separatori supportati: virgola, punto e virgola, slash, pipe ripetuta, newline
        $vals = preg_split('/[,;\\/|\\r\\n]+/u', $values_str);
        $clean = [];
        foreach ($vals as $v) {
            $v = trim($v);
            if ($v !== '') { $clean[$v] = true; }
        }
        if (!empty($clean)) {
            $pairs[$label] = array_keys($clean);
        }
    }
    return $pairs;
}

?>