
<?php
/**
 * Import asincrono a lotti con stato persistito e progress JSON.
 * Dipendenze: danea_imp_prodotti.php (helper ensureCategoryPS9_cached, extractFeaturePairsFromCustomFields)
 */
if (!defined('_PS_VERSION_')) { exit; }
@ini_set('display_errors', '0');

function spb_danea_async_get_xml_path()
{
    $xml_path = (string)Configuration::get('SB_DANEA_LAST_XML');
    if (!$xml_path) {
        $xml_path = _PS_MODULE_DIR_ . 'spiderbyte_danea/tmp/danea_last.xml';
    }
    return $xml_path;
}

function spb_danea_async_state_key() { return 'SB_DANEA_IMPORT_STATE'; }

/**
 * Raccoglie i nodi Product sia da <Products> che da <UpdatedProducts>.
 * Ritorna array di SimpleXMLElement (anche vuoto).
 */
function spb_danea_collect_product_nodes(SimpleXMLElement $xml)
{
    $productNodes = [];

    // 1) complete export: <Products><Product>...</Product></Products>
    if (isset($xml->Products) && isset($xml->Products->Product)) {
        foreach ($xml->Products->Product as $p) { $productNodes[] = $p; }
    }

    // 2) incremental export: <UpdatedProducts><Product>...</Product></UpdatedProducts>
    if (empty($productNodes)) {
        $tmp = $xml->xpath('//UpdatedProducts/Product');
        if (is_array($tmp)) { $productNodes = $tmp; }
    }

    // 3) Fallback combinato (robustezza)
    if (empty($productNodes)) {
        $tmp = $xml->xpath('//Products/Product | //UpdatedProducts/Product');
        if (is_array($tmp)) { $productNodes = $tmp; }
    }

    return $productNodes;
}

function spb_danea_async_start(Module $module, Context $ctx, $batch)
{
    $xml_path = spb_danea_async_get_xml_path();
    if (!file_exists($xml_path)) {
        return array('ok'=>false,'msg'=>'XML non trovato','path'=>$xml_path);
    }

    libxml_use_internal_errors(true);
    $xml = simplexml_load_file($xml_path);
    if (!$xml) {
        return array('ok'=>false,'msg'=>'XML non leggibile');
    }

    // === PATCH: conteggio prodotti robusto (Products e/o UpdatedProducts) ===
    $nodes = spb_danea_collect_product_nodes($xml);
    $total = is_array($nodes) ? count($nodes) : (int)$nodes;

    if ($total <= 0) {
        return array('ok'=>false,'msg'=>'Nessun prodotto nell\'XML');
    }
    // === FINE PATCH ===

    $id_trg = (int)Configuration::get('SB_DANEA_TAX_RULE_GROUP_ID');
    if ($id_trg <= 0) {
        return array('ok'=>false,'msg'=>'Seleziona un gruppo IVA nel modulo (SB_DANEA_TAX_RULE_GROUP_ID).');
    }

    $state = array(
        'started_at' => date('c'),
        'xml_path'   => $xml_path,
        'batch'      => max(1, (int)$batch),
        'total'      => (int)$total,
        'offset'     => 0,
        'created'    => 0, 'updated' => 0,
        'assoc_cat'  => 0, 'assoc_brand' => 0, 'assoc_feat' => 0,
        'qty_set'    => 0, 'ean_set' => 0, 'price_set' => 0, 'tax_set' => 0,
        'done'       => false,
    );
    Configuration::updateValue(spb_danea_async_state_key(), json_encode($state));

    return array('ok'=>true,'msg'=>'Init OK','state'=>$state);
}

function spb_danea_async_run(Module $module, Context $ctx)
{
    $raw = Configuration::get(spb_danea_async_state_key());
    if (!$raw) { return array('ok'=>false,'msg'=>'Stato non inizializzato'); }
    $state = json_decode($raw, true);
    if (!$state || !empty($state['done'])) {
        return array('ok'=>false,'msg'=>'Import già completato o stato non valido');
    }

    $xml_path = $state['xml_path'];
    if (!file_exists($xml_path)) { return array('ok'=>false,'msg'=>'XML non trovato'); }

    libxml_use_internal_errors(true);
    $xml = simplexml_load_file($xml_path);
    if (!$xml) { return array('ok'=>false,'msg'=>'XML non leggibile'); }

    $logic = _PS_MODULE_DIR_.$module->name.'/controllers/admin/danea_imp_prodotti.php';
    if (!file_exists($logic)) { return array('ok'=>false,'msg'=>'danea_imp_prodotti.php mancante'); }
    include_once($logic);

    $id_lang = 1;
    $id_shop = (int)$ctx->shop->id;
    $id_home = 2;
    $id_trg  = (int)Configuration::get('SB_DANEA_TAX_RULE_GROUP_ID');

    $cache_categories = [];
    $cache_features   = [];
    $cache_feat_vals  = [];
    $cache_brands     = [];

    $offset = (int)$state['offset'];
    $limit  = (int)$state['batch'];
    $total  = (int)$state['total'];

    // === PATCH: raccogli lista prodotti da entrambe le sezioni ===
    $productNodes = spb_danea_collect_product_nodes($xml);
    if (empty($productNodes)) {
        return array('ok'=>false,'msg'=>'Nessun prodotto nell\'XML');
    }
    // === FINE PATCH ===

    $stop = min($total, $offset + $limit);

    for ($idx = $offset; $idx < $stop; $idx++) {
        $prod = $productNodes[$idx];

        $code      = trim((string)$prod->Code);
        $ean       = trim((string)$prod->Barcode);
        $iid       = trim((string)$prod->InternalID);
        $reference = $code !== '' ? $code : ($ean !== '' ? $ean : $iid);
        if ($reference === '') { continue; }

        $id_product = (int)Db::getInstance()->getValue(
            'SELECT id_product FROM '._DB_PREFIX_.'product WHERE reference = "'.pSQL($reference).'"'
        );

        if ($id_product) {
            $product = new Product($id_product, false, $id_lang);
            $state['updated']++;
        } else {
            $name = trim((string)$prod->Description);
            if ($name === '') { $name = $reference; }
            $slug = Tools::str2url($name);

            $product = new Product();
            $product->active = 1;
            $product->reference = $reference;
            $product->id_category_default = $id_home;
            $product->name = [$id_lang => $name];
            $product->link_rewrite = [$id_lang => $slug];
            $product->price = 0;
            $product->id_tax_rules_group = (int)$id_trg;
            $product->add();
            $id_product = (int)$product->id;

            $state['created']++;
            $state['tax_set']++;
        }

        $dirty = false;

        // Brand / Manufacturer
        $brand_name = isset($prod->ProducerName) ? trim((string)$prod->ProducerName) : '';
        if ($brand_name !== '') {
            if (!isset($cache_brands[$brand_name])) {
                $id_man = (int)Db::getInstance()->getValue(
                    'SELECT id_manufacturer FROM '._DB_PREFIX_.'manufacturer WHERE name = "'.pSQL($brand_name).'"'
                );
                if (!$id_man) {
                    $m = new Manufacturer();
                    $m->name = $brand_name;
                    $m->active = 1;
                    $m->add();
                    $id_man = (int)$m->id;
                }
                $cache_brands[$brand_name] = $id_man;
            }
            $id_manufacturer = (int)$cache_brands[$brand_name];
            if ((int)$product->id_manufacturer !== $id_manufacturer) {
                $product->id_manufacturer = $id_manufacturer;
                $dirty = true;
                $state['assoc_brand']++;
            }
        }

        // Categorie
        $cat_name = trim((string)$prod->Category);
        $sub_name = isset($prod->Subcategory) ? trim((string)$prod->Subcategory) : '';
        $ids_to_add = [];
        if ($cat_name !== '') {
            $id_cat = ensureCategoryPS9_cached($cat_name, $id_home, $id_lang, $id_shop, $cache_categories);
            if ($id_cat) { $ids_to_add[] = $id_cat; }
            if ($sub_name !== '') {
                $id_sub = ensureCategoryPS9_cached($sub_name, $id_cat, $id_lang, $id_shop, $cache_categories);
                if ($id_sub) { $ids_to_add[] = $id_sub; }
                $default_cat = $id_sub;
            } else {
                $default_cat = $id_cat;
            }
        } else {
            $default_cat = $id_home;
        }

        if ((int)$product->id_category_default !== (int)$default_cat) {
            $product->id_category_default = (int)$default_cat;
            $dirty = true;
        }

        if (!empty($ids_to_add)) {
            $existing = Db::getInstance()->executeS('SELECT id_category FROM '._DB_PREFIX_.'category_product WHERE id_product='.(int)$id_product);
            $have = [];
            foreach ($existing as $row) { $have[(int)$row['id_category']] = true; }
            foreach ($ids_to_add as $cid) {
                if (!isset($have[(int)$cid])) {
                    Db::getInstance()->insert('category_product', [
                        'id_category' => (int)$cid,
                        'id_product'  => (int)$id_product,
                        'position'    => 0,
                    ]);
                    $state['assoc_cat']++;
                }
            }
        }

        // Prezzo netto (Listino 1)
        if (isset($prod->NetPrice1)) {
            $net = (float)str_replace(',', '.', (string)$prod->NetPrice1);
            if ($net < 0) { $net = 0; }
            if ((float)$product->price !== $net) {
                $product->price = $net;
                $dirty = true;
                $state['price_set']++;
            }
        }

        // Gruppo IVA
        if ((int)$product->id_tax_rules_group !== (int)$id_trg) {
            $product->id_tax_rules_group = (int)$id_trg;
            $dirty = true;
            $state['tax_set']++;
        }

        if ($dirty) {
            $product->update();
            $dirty = false;
        }

        // EAN/Barcode
        $barcode_raw = isset($prod->Barcode) ? (string)$prod->Barcode : '';
        $barcode = trim($barcode_raw);
        if ($barcode !== '') {
            Db::getInstance()->update('product', ['ean13' => pSQL($barcode)], 'id_product='.(int)$id_product);
            $state['ean_set']++;
        }

        // Quantità
        if (isset($prod->AvailableQty)) {
            $qty_raw = trim((string)$prod->AvailableQty);
            $qty = (int)floor((float)str_replace(',', '.', $qty_raw));
            if ($qty < 0) { $qty = 0; }
            StockAvailable::setQuantity((int)$product->id, 0, (int)$qty, (int)$id_shop);
            $state['qty_set']++;
        }

        // Caratteristiche da campi custom
        $pairs = extractFeaturePairsFromCustomFields($prod);
        if (!empty($pairs)) {
            foreach ($pairs as $label => $values) {
                if (!isset($cache_features[$label])) {
                    $id_feature = (int)Db::getInstance()->getValue(
                        'SELECT f.id_feature FROM '._DB_PREFIX_.'feature f '
                        . 'INNER JOIN '._DB_PREFIX_.'feature_lang fl ON (fl.id_feature=f.id_feature AND fl.id_lang='.(int)$id_lang.') '
                        . 'WHERE fl.name = "'.pSQL($label).'"'
                    );
                    if (!$id_feature) {
                        $f = new Feature();
                        $f->position = 0;
                        $f->name = [$id_lang => $label];
                        $f->add();
                        $id_feature = (int)$f->id;
                    }
                    $cache_features[$label] = $id_feature;
                }
                $id_feature = (int)$cache_features[$label];

                foreach ($values as $value) {
                    if ($value === '') { continue; }
                    $kv = $label."\n\n\n".$value;
                    if (!isset($cache_feat_vals[$kv])) {
                        $id_val = (int)Db::getInstance()->getValue(
                            'SELECT fv.id_feature_value FROM '._DB_PREFIX_.'feature_value fv '
                            . 'INNER JOIN '._DB_PREFIX_.'feature_value_lang fvl ON (fvl.id_feature_value=fv.id_feature_value AND fvl.id_lang='.(int)$id_lang.') '
                            . 'WHERE fv.id_feature='.(int)$id_feature.' AND fvl.value = "'.pSQL($value).'"'
                        );
                        if (!$id_val) {
                            $fv = new FeatureValue();
                            $fv->id_feature = $id_feature;
                            $fv->custom = 0;
                            $fv->value = [$id_lang => $value];
                            $fv->add();
                            $id_val = (int)$fv->id;
                        }
                        $cache_feat_vals[$kv] = $id_val;
                    }
                    $id_value = (int)$cache_feat_vals[$kv];

                    Db::getInstance()->delete('feature_product', 'id_product='.(int)$id_product.' AND id_feature='.(int)$id_feature);
                    Db::getInstance()->insert('feature_product', [
                        'id_feature'       => (int)$id_feature,
                        'id_product'       => (int)$id_product,
                        'id_feature_value' => (int)$id_value,
                    ]);
                    $state['assoc_feat']++;
                }
            }
        }
    }

    $state['offset'] = $stop;
    if ($state['offset'] >= $state['total']) {
        $state['done'] = true;
    }
    Configuration::updateValue(spb_danea_async_state_key(), json_encode($state));

    $pct = max(0, min(100, $state['total'] > 0 ? round($state['offset']*100/$state['total']) : 0));
    return array('ok'=>true,'progress'=>$pct,'state'=>$state);
}

function spb_danea_async_status()
{
    $raw = Configuration::get(spb_danea_async_state_key());
    if (!$raw) { return array('ok'=>false,'msg'=>'Nessuno stato'); }
    $state = json_decode($raw, true);
    $pct = max(0, min(100, $state['total'] > 0 ? round($state['offset']*100/$state['total']) : 0));
    return array('ok'=>true,'progress'=>$pct,'state'=>$state);
}

function spb_danea_async_cancel()
{
    Configuration::updateValue(spb_danea_async_state_key(), '');
    return array('ok'=>true);
}
